## Alcohol Use data: Analysis using mixed regression models -------------------
## Author: Ricardo Rasmussen Petterle DMI/UFPR --------------------------------
## Date: June 29, 2023 ---------------------------------------------------------

rm(list=ls())

## BEGIN HEADER ----------------------------------------------------------------
## Loading extra packages
library(zoib)
library(gamlss)
library(lattice)
library(latticeExtra)
library(TMB)
library(ggplot2)
library(tidyverse)
library(cowplot)

## Loading extra function
source("GOF.R")

## Loading data set
data("AlcoholUse", package = "zoib")
da <- AlcoholUse

## Preparing data set 
da$County <- factor(da$County)

da$Grade <- factor(da$Grade)
levels(da$Grade)

da$Gender <- factor(da$Gender)
levels(da$Gender)

da$MedDays <- factor(da$MedDays)
levels(da$MedDays)
## END HEADER -----------------------------------------------------------------

## GAMLSS for initial values (model 1)
fit_gamlss1 <- gamlss(Percentage ~ Grade * Gender + MedDays + re(random=~1|County),
                     family = BEZI,
                      data = da)
summary(fit_gamlss1)

beta1 = as.numeric(fit_gamlss1$mu.coefficients)[-8]
phi1 = as.numeric(fit_gamlss1$sigma.coefficients)
gamma1 = as.numeric(fit_gamlss1$nu.coefficients)

## Fitting using Template Model Builder (TMB) 
## Model 1: ZAUG --------------------------------------------------------------
# compile("Unit_Gamma_Zero_Inflated_Mixed.cpp")
dyn.load(dynlib("Unit_Gamma_Zero_Inflated_Mixed"))

X <- model.matrix(~ Grade * Gender + MedDays, da)
Z <- model.matrix(~ County-1, da)
S <- model.matrix(~ 1, da)
X <- as(X,"dgTMatrix")
Z <- as(Z,"dgTMatrix")
S <- as(S,"dgTMatrix")
u <- rnorm(ncol(Z))
da$y <- da$Percentage

## Fit model
UG1 <- MakeADFun(data = list(y = da$y, X = X, Z = Z, S = S),
                parameters = list(beta = beta1, gamma = gamma1,
                                  logphi = phi1, logsdu = 1, u = u*0),
                random = "u",
                hessian = T,
                DLL = "Unit_Gamma_Zero_Inflated_Mixed",
                silent = FALSE)
## Optimization
opt_UG1 <- nlminb(UG1$par, UG1$fn, UG1$gr)
opt_UG1
model1 <- sdreport(UG1)
summary(model1, "fixed", p.value = T)
summary(model1, "report", p.value = T)

## Goodness-of-fit measures
GOF(model_opt = opt_UG1, N = 1340)

## Model 1: ZAB ---------------------------------------------------------------
# compile("Beta_Zero_Inflated_Mixed.cpp")
dyn.load(dynlib("Beta_Zero_Inflated_Mixed"))

## Fit model
Beta1 <- MakeADFun(data = list(y = da$y, X = X, Z = Z, S = S),
                  parameters = list(beta = beta1, gamma = gamma1,
                                    logphi = phi1, logsdu = 1, u = u*0),
                  random = "u",
                  hessian = T,
                  DLL = "Beta_Zero_Inflated_Mixed",
                  silent = FALSE)
## Optimization
opt_beta1 <- nlminb(Beta1$par, Beta1$fn, Beta1$gr)
opt_beta1
model1 <- sdreport(Beta1)
summary(model1, "fixed", p.value = T)
summary(model1, "report", p.value = T)

## Goodness-of-fit measures
GOF(model_opt = opt_beta1, N = 1340)

## GAMLSS for initial values (model 2) ----------------------------------------
fit_gamlss2 <- gamlss(Percentage ~ Grade * Gender + MedDays + re(random=~1|County), 
                      nu.formula = ~ Grade * Gender + MedDays,
                      family = BEZI,
                      data = da)
summary(fit_gamlss2)

beta2 = as.numeric(fit_gamlss2$mu.coefficients)[-8]
phi2 = as.numeric(fit_gamlss2$sigma.coefficients)
gamma2 = as.numeric(fit_gamlss2$nu.coefficients)

## Model 2: ZAUG --------------------------------------------------------------
# compile("Unit_Gamma_Zero_Inflated_Mixed.cpp")
dyn.load(dynlib("Unit_Gamma_Zero_Inflated_Mixed"))

X <- model.matrix(~ Grade * Gender + MedDays, da)
Z <- model.matrix(~ County-1, da)
S <- model.matrix(~ Grade * Gender + MedDays, da)
X <- as(X,"dgTMatrix")
Z <- as(Z,"dgTMatrix")
S <- as(S,"dgTMatrix")
u <- rnorm(ncol(Z))
da$y <- da$Percentage

## Fit model
UG2 <- MakeADFun(data = list(y = da$y, X = X, Z = Z, S = S),
                parameters = list(beta = beta2, gamma = gamma2,
                                  logphi = phi2, logsdu = 1, u = u*0),
                random = "u",
                hessian = T,
                DLL = "Unit_Gamma_Zero_Inflated_Mixed",
                silent = FALSE)
## Optimization
opt_UG2 <- nlminb(UG2$par, UG2$fn, UG2$gr)
opt_UG2
model2 <- sdreport(UG2)
summary(model2, "fixed", p.value = TRUE)
summary(model2, "report", p.value = TRUE)

## Goodness-of-fit measures
GOF(model_opt = opt_UG2, N = 1340)

## Model 2: ZAB ---------------------------------------------------------------
# compile("Beta_Zero_Inflated_Mixed.cpp")
dyn.load(dynlib("Beta_Zero_Inflated_Mixed"))

## Fit model
Beta2 <- MakeADFun(data = list(y = da$y, X = X, Z = Z, S = S),
                  parameters = list(beta = beta2, gamma = gamma2,
                                    logphi = phi2, logsdu = 1, u = u*0),
                  random = "u",
                  hessian = TRUE,
                  DLL = "Beta_Zero_Inflated_Mixed",
                  silent = FALSE)
## Optimization
opt_beta2 <- nlminb(Beta2$par, Beta2$fn, Beta2$gr)
opt_beta2
model2 <- sdreport(Beta2)
summary(model2, "fixed", p.value = TRUE)
summary(model2, "report", p.value = TRUE)

## Goodness-of-fit measures
GOF(model_opt = opt_beta2, N = 1340)

## Comparing ------------------------------------------------------------------

## Log-likelihood
cbind("M1:ZAUG" = abs(opt_UG1$objective), "M1:ZAB" = abs(opt_beta1$objective),
      "M2:ZAUG" = abs(opt_UG2$objective), "M2:ZAB" = abs(opt_beta2$objective))
## END ------------------------------------------------------------------------
